
module environment_model_inmcm

    !< @brief дополнительный модуль только для модели INMCM
    !< @details дублирует код деятельного слоя, необходимый для модели углерода, благодаря чему позволяет разделить их программную реализацию

    ! интерфейс
    ! -------------------------------------------------------------------------------------------------------------------
    
    use grid, only : ml
    use environment_core, only : rhodry
    implicit none

    public

    ! переменные
    ! -------------------------------------------------------------------------------------------------------------------
    
    integer, parameter :: nv2 = 13  ! временное локальное предписанное значение
    integer, parameter :: ns2 = 9

!    real, parameter :: rhodry(ml) = 1.0  !< плотность сухой почвы, г/см3
	                                ! 1.0  DAO 0.7 1.26
									! 2.62 Rostov 1.28
    !real :: g2gw_to_cmw(ml)                !< г/г воды в см воды в слоях почвы

    real :: sncr = 0.4
    real :: sq(4)                          !< Fractions covered by different land surface types
    real :: weight(nv2)                    !< веса типов растительности
    
    real, parameter :: rootsm(nv2) = (/100.,100.,100.,100.,50.,100.,50.,50.,50.,50.,50.,50.,50./)
    real, parameter :: psi1(nv2) = (/-100.,-190.,-200.,-200.,-200.,-190.,-120.,-200.,-200.,-200.,-0.30,-190.,-100./)
    real, parameter :: psi2(nv2) = (/-500.,-250.,-250.,-250.,-250.,-250.,-230.,-400.,-400.,-400.,-150.,-250.,-200./)

    real :: wssg(ml,nv2)                         !< водный потенциал начала завядания
    real :: wssl(ml,nv2)                         !< водный потенциал полного завядания
    real :: roots(ml,nv2)                        !< профиль плотности корневой системы
    
    real, parameter :: dtl(nv2) = (/4.5, 4.0, 5.0, 7.0, 2.0, 3.0, 3.0, 0.8, 1.0, 1.5, 0.4, 0.5, 3.0/)
    real :: amnvv(nv2)
    
    real :: gai(nv2,12)
    !               Jan  Feb  Mar  Apr  May  Jun  Jul  Aug  Sep  Oct  Nov  Dec
    data gai(01,:) /4.5, 4.5, 4.5, 4.5, 4.5, 4.5, 4.5, 4.5, 4.5, 4.5, 4.5, 4.5/
    data gai(02,:) /0.0, 0.0, 0.3, 1.2, 3.0, 4.7, 4.5, 3.4, 1.2, 0.3, 0.0, 0.0/
    data gai(03,:) /0.4, 0.4, 0.2, 0.6, 0.9, 1.4, 2.2, 2.4, 1.8, 1.1, 0.6, 0.5/
    data gai(04,:) /4.1, 4.2, 4.6, 4.8, 4.9, 5.0, 4.8, 4.7, 4.6, 4.2, 4.0, 4.0/
    data gai(05,:) /0.0, 0.0, 0.0, 0.6, 1.2, 2.0, 2.6, 1.7, 1.0, 0.5, 0.2, 0.0/
    data gai(06,:) /0.8, 0.7, 0.4, 0.5, 0.5, 0.7, 1.7, 3.0, 2.5, 1.6, 1.0, 1.0/
    data gai(07,:) /0.4, 0.5, 0.6, 0.7, 1.2, 3.0, 3.5, 1.5, 0.7, 0.6, 0.5, 0.4/
    data gai(08,:) /1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0/
    data gai(09,:) /0.9, 0.8, 0.2, 0.2, 0.0, 0.0, 0.0, 0.2, 0.4, 0.5, 0.6, 0.8/
    data gai(10,:) /0.0, 0.0, 0.0, 0.0, 0.0, 0.2, 1.4, 1.2, 0.0, 0.0, 0.0, 0.0/
    data gai(11,:) /0.4, 0.5, 0.6, 0.7, 1.2, 3.0, 3.5, 1.5, 0.7, 0.6, 0.5, 0.4/
    data gai(12,:) /0.0, 0.0, 0.0, 0.0, 1.0, 2.0, 3.0, 3.0, 1.5, 0.0, 0.0, 0.0/
    data gai(13,:) /0.0, 0.0, 0.0, 0.0, 1.0, 2.0, 3.0, 3.0, 1.5, 0.0, 0.0, 0.0/

    real :: sai(nv2,12)
    !               Jan  Feb  Mar  Apr  May  Jun  Jul  Aug  Sep  Oct  Nov  Dec
    data sai(01,:) /0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5/
    data sai(02,:) /0.4, 0.4, 0.4, 0.4, 0.5, 0.4, 0.9, 1.4, 2.6, 1.4, 0.6, 0.4/
    data sai(03,:) /0.4, 0.3, 0.4, 0.3, 0.3, 0.3, 1.0, 0.9, 0.8, 0.9, 0.8, 0.4/
    data sai(04,:) /0.4, 0.5, 0.4, 0.3, 0.4, 0.5, 0.5, 0.6, 0.6, 0.7, 0.6, 0.5/
    data sai(05,:) /0.3, 0.3, 0.3, 0.4, 0.4, 0.4, 1.7, 1.2, 1.0, 0.8, 0.6, 0.5/
    data sai(06,:) /0.4, 0.3, 0.5, 0.3, 0.3, 0.3, 0.3, 0.7, 0.7, 1.1, 0.9, 0.2/
    data sai(07,:) /0.3, 0.3, 0.3, 0.3, 0.3, 0.4, 0.8, 2.3, 1.1, 0.4, 0.4, 0.4/
    data sai(08,:) /0.3, 0.3, 0.3, 0.3, 0.3, 0.3, 0.3, 0.3, 0.3, 0.3, 0.3, 0.3/
    data sai(09,:) /0.1, 0.2, 0.6, 0.1, 0.6, 0.0, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1/
    data sai(10,:) /0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.5, 1.4, 0.1, 0.1, 0.1/
    data sai(11,:) /0.3, 0.3, 0.3, 0.3, 0.3, 0.4, 0.8, 2.3, 1.1, 0.4, 0.4, 0.4/
    data sai(12,:) /0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0/
    data sai(13,:) /0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0/

    real :: tai(nv2,12)
    !               Jan  Feb  Mar  Apr  May  Jun  Jul  Aug  Sep  Oct  Nov  Dec
    data tai(01,:) /5.0, 5.0, 5.0, 5.0, 5.0, 5.0, 5.0, 5.0, 5.0, 5.0, 5.0, 5.0/
    data tai(02,:) /0.4, 0.4, 0.7, 1.6, 3.5, 5.1, 5.4, 4.8, 3.8, 1.7, 0.6, 0.4/
    data tai(03,:) /0.8, 0.7, 0.6, 0.9, 1.2, 1.7, 3.2, 3.3, 2.6, 2.0, 1.4, 0.9/
    data tai(04,:) /4.5, 4.7, 5.0, 5.1, 5.3, 5.5, 5.3, 5.3, 5.2, 4.9, 4.6, 4.5/
    data tai(05,:) /0.3, 0.3, 0.3, 1.0, 1.6, 2.4, 4.3, 2.9, 2.0, 1.3, 0.8, 0.5/
    data tai(06,:) /1.2, 1.0, 0.9, 0.8, 0.8, 1.0, 2.0, 3.7, 3.2, 2.7, 1.9, 1.2/
    data tai(07,:) /0.7, 0.8, 0.9, 1.0, 1.5, 3.4, 4.3, 3.8, 1.8, 1.0, 0.9, 0.8/
    data tai(08,:) /1.3, 1.3, 1.3, 1.3, 1.3, 1.3, 1.3, 1.3, 1.3, 1.3, 1.3, 1.3/
    data tai(09,:) /1.0, 1.0, 0.8, 0.3, 0.6, 0.0, 0.1, 0.3, 0.5, 0.6, 0.7, 0.9/
    data tai(10,:) /0.1, 0.1, 0.1, 0.1, 0.1, 0.3, 1.5, 1.7, 1.4, 0.1, 0.1, 0.1/
    data tai(11,:) /0.7, 0.8, 0.9, 1.0, 1.5, 3.4, 4.3, 3.8, 1.8, 1.0, 0.9, 0.8/
    data tai(12,:) /0.0, 0.0, 0.0, 0.0, 1.0, 2.0, 3.0, 3.0, 1.5, 0.0, 0.0, 0.0/
    data tai(13,:) /0.0, 0.0, 0.0, 0.0, 1.0, 2.0, 3.0, 3.0, 1.5, 0.0, 0.0, 0.0/

    real, target :: tlai(nv2), tsai(nv2), elai(nv2), esai(nv2), vai(nv2)
    real :: fsun(nv2), fsha(nv2)
    
    real :: htop(nv2)  ! top of canopy (m)
    data htop(:) /35.00, 20.00, 17.00, 17.00, 14.00, 18.00, 0.50, 0.50, 0.50, 0.50, 0.50, 0.50, 0.50/
    real :: hbot(nv2)  ! bottom of canopy (m)
    data hbot(:) / 1.00, 11.50,  9.25,  8.50,  7.00, 10.00, 0.01, 0.10, 0.10, 0.10, 0.01, 0.01, 0.01/

    ! albedo
    integer, parameter :: twoband = 2
    ! omega,betad,betai for snow
    real, parameter :: omegas(twoband) = (/0.8, 0.4/)
    real, parameter :: betads = 0.5              !< Betad for snow
    real, parameter :: betais = 0.5             !< Betai for snow

    real :: xl(nv2)  ! leaf/stem orientation index: valid range = -0.4 to 0.6
    ! departure of leaf angles from spherical distribution
    data xl(:) /0.10, 0.25, 0.13, 0.01, 0.01, 0.01,-0.30,0.01, 0.25, 0.25,-0.30,-0.30,-0.30/

    real :: rhol(twoband,nv2), rhos(twoband,nv2), taul(twoband,nv2), taus(twoband,nv2)
    ! leaf reflectance: 1=vis, 2=nir
    data rhol(1,:) /0.10, 0.10, 0.09, 0.07, 0.07, 0.10, 0.11, 0.07, 0.10, 0.10, 0.11, 0.11, 0.11/
    data rhol(2,:) /0.45, 0.45, 0.40, 0.35, 0.35, 0.45, 0.58, 0.35, 0.45, 0.45, 0.58, 0.58, 0.58/

    ! stem reflectance: 1=vis, 2=nir
    data rhos(1,:) /0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.36, 0.16, 0.16, 0.16, 0.36, 0.36, 0.36/
    data rhos(2,:) /0.39, 0.39, 0.39, 0.39, 0.39, 0.39, 0.58, 0.39, 0.39, 0.39, 0.58, 0.58, 0.58/

    ! leaf transmittance: 1=vis, 2=nir
    data taul(1,:) /0.05, 0.05, 0.05, 0.05, 0.05, 0.05, 0.07, 0.05, 0.05, 0.05, 0.07, 0.07, 0.07/
    data taul(2,:) /0.25, 0.25, 0.18, 0.10, 0.10, 0.25, 0.25, 0.10, 0.25, 0.25, 0.25, 0.25, 0.25/

    ! stem transmittance: 1=vis, 2=nir
    data taus(1,:) /0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.220, 0.001, 0.001, 0.001, 0.220, 0.220, 0.220/
    data taus(2,:) /0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.380, 0.001, 0.001, 0.001, 0.380, 0.380, 0.380/

    real :: albsat(9,twoband), albdry(9,twoband)
    ! saturated soil albedos: 1=vis, 2=nir
    data albsat(:,1) /0.12, 0.11, 0.10, 0.09, 0.08, 0.07, 0.06, 0.05, 0.22/
    data albsat(:,2) /0.24, 0.22, 0.20, 0.18, 0.16, 0.14, 0.12, 0.10, 0.34/
    ! dry soil albedos: 1=vis, 2=nir
    data albdry(:,1) /0.24, 0.22, 0.20, 0.18, 0.16, 0.14, 0.12, 0.10, 0.34/
    data albdry(:,2) /0.48, 0.44, 0.40, 0.36, 0.32, 0.28, 0.24, 0.20, 0.58/
    
    real, allocatable :: cosz(:,:)
    integer, allocatable :: olim(:,:)
    real :: fwet(nv2)

    integer, parameter :: idir = 1, idif = 2
    integer, parameter :: ivis = 1, inir = 2

    real, allocatable :: vegg_nv2(:,:,:), vegg_nv1(:,:,:)
    real, allocatable :: soilcol(:,:,:), soilcol_wide(:,:,:)
    real, allocatable :: sPORu(:,:), sBHu(:,:), sPSIMAXu(:,:), wssl_glob(:,:,:,:), wssg_glob(:,:,:,:)

    real, allocatable :: MCD15A2H_LAI(:,:,:,:)


contains

    ! внешние процедуры
    ! -------------------------------------------------------------------------------------------------------------------

    subroutine environment_model_init()
        ! ---------------------------------------
        use config, only : nv_singlecolumn
        use const, only : miss_v, rho_w, m2cm
        use paths, only : path_inmcm_data
        use grid, only : ms, dz, z, i0, i1, j0, j1, nlon, nlat, nich, ich_to_i, ich_to_j
        use netcdf
        use netcdf_kit, only : nc_errhand
        integer :: i, j, k, n, ncid, varid
        integer, allocatable :: work1i(:)
        
        allocate(vegg_nv1(nlon,nlat,nv2))
        allocate(vegg_nv2(i0:i1,j0:j1,nv2))
        allocate(soilcol_wide(nlon,nlat,ns2))
        allocate(soilcol(i0:i1,j0:j1,ns2))
        allocate(sPORu(nlon,nlat))
        allocate(sBHu(nlon,nlat))
        allocate(sPSIMAXu(nlon,nlat))
        allocate(wssl_glob(i0:i1,j0:j1,ml,nv2))
        allocate(wssg_glob(i0:i1,j0:j1,ml,nv2))
        allocate(cosz(i0:i1,j0:j1))
        
        vegg_nv1(:,:,:) = miss_v
        open(1, file=path_inmcm_data//'VEG_BART', status='old')
        allocate(work1i(nich))
        do n = 1, nv2
            read(1,*)
            read(1,'(24i3)') work1i
            do i = 1, nich
                vegg_nv1(ich_to_i(i), ich_to_j(i), n) = float(work1i(i)) * 0.01
            enddo
        enddo
        deallocate(work1i)
        close(1)

        soilcol_wide(:,:,:) = miss_v
        open(1, file=path_inmcm_data//'SOILCOL', status='old')
        allocate(work1i(nich))
        do n = 1, ns2
            read(1,*)
            read(1,'(24i3)') work1i
            do i = 1, nich
                soilcol_wide(ich_to_i(i), ich_to_j(i), n) = float(work1i(i)) * 0.01
            enddo
        enddo
        deallocate(work1i)
        close(1)
        
       ! call special_soilpar_read(path_inmcm_data//'soil_par_average.nc', 'theta_s', sPORu, fill_v = 0.4)
       ! call special_soilpar_read(path_inmcm_data//'CH_par_average.nc', 'psi_s', sPSIMAXu, fill_v = -17.)
       ! call special_soilpar_read(path_inmcm_data//'CH_par_average.nc', 'lambda', sBHu, fill_v = 0.15)
		sPORu = 0.4
		sPSIMAXu = -17.
		sBHu = 0.15
        sBHu(:,:) = 1./sBHu(:,:)
        
        allocate(MCD15A2H_LAI(720,360,nv2,12))
       ! call nc_errhand( nf90_open(path_inmcm_data//'mcd15a2h_to_inmcm.nc', nf90_nowrite, ncid) )
       ! call nc_errhand( nf90_inq_varid(ncid, 'lai', varid) )
       ! call nc_errhand( nf90_get_var(ncid, varid, MCD15A2H_LAI(:,:,:,:), (/1,1,1,1/), (/720,360,nv2,12/)) )
       ! call nc_errhand( nf90_close(ncid) )
		MCD15A2H_LAI = 2.
        
        if (nv_singlecolumn /= miss_v) then
            vegg_nv1(:,:,:) = 0.
            vegg_nv1(:,:,nv_singlecolumn) = 1.  ! Федоровское
        endif
        
        amnvv(1:9) = 1.
        amnvv(10) = dtl(10)/(dtl(10)+dtl(11))
        amnvv(11) = dtl(11)/(dtl(10)+dtl(11))
        amnvv(12) = dtl(12)/(dtl(12)+dtl(13))
        amnvv(13) = dtl(13)/(dtl(12)+dtl(13))
        
        vegg_nv2(:,:,1:9) = vegg_nv1(i0:i1,j0:j1,1:9)
        vegg_nv2(:,:,10) = vegg_nv1(i0:i1,j0:j1,10) * amnvv(10)
        vegg_nv2(:,:,11) = vegg_nv1(i0:i1,j0:j1,10) * amnvv(11)
        vegg_nv2(:,:,12) = vegg_nv1(i0:i1,j0:j1,12) * amnvv(12)
        vegg_nv2(:,:,13) = vegg_nv1(i0:i1,j0:j1,12) * amnvv(13)
        
        soilcol(:,:,:) = soilcol_wide(i0:i1,j0:j1,:)
        
        !g2gw_to_cmw(:) = miss_v
       ! do k = ms+1, ml-1
       !     g2gw_to_cmw(k) = rhodry(k) / rho_w * dz(k)
       ! end do
        
        do n = 1, nv2
            do k = ms+1, ml-1
                if (z(k) < 10.) then
                    roots(k,n) = 5.
                elseif (z(k) < rootsm(n)) then
                    roots(k,n) = 1.
                else
                    roots(k,n) = 0.
                endif
            end do
        end do
        
        do i = i0, i1
            do j = j0, j1
                do k = ms+1, ml-1
                    do n = 1, nv2
                        wssl_glob(i,j,k,n) = sPORu(i,j)*(psi2(n)*m2cm/sPSIMAXu(i,j))**(-1./sBHu(i,j))/rhodry(i,j,k)
                        wssg_glob(i,j,k,n) = sPORu(i,j)*(psi1(n)*m2cm/sPSIMAXu(i,j))**(-1./sBHu(i,j))/rhodry(i,j,k)
                    enddo
                enddo
            enddo
        enddo
        
    end subroutine


    subroutine environment_model_calc_at_timestep()
        ! ---------------------------------------
        call sunec()
    
    end subroutine
    
    
    subroutine environment_model_calc_at_cell(ii,jj)
        ! ---------------------------------------
        use environment_core, only : snow
        integer, intent(in) :: ii, jj

        sq(4) = min(0.99,snow(ii,jj)/sncr)
        
    end subroutine
    
    
    subroutine environment_model_calc_at_tile(ii,jj,nn)
        ! ---------------------------------------
        use grid, only : date_c
        use environment_core, only : snow
        
        integer, intent(in) :: ii, jj, nn
        real :: ol, fb
        
        weight(nn) = vegg_nv2(ii,jj,nn)*(1.0-sq(4))
        fwet(nn) = 0.! 0.5
        
        tlai(nn) = gai(nn,1) + amnvv(nn)*(gai(nn,7) - gai(nn,1))
        tsai(nn) = tai(nn,1) + amnvv(nn)*(tai(nn,7) - tai(nn,1))
        ! tlai(nn)=gai(nn,date_c%m)*amnvv(nn)                       ! land_snow
        ! tsai(nn)=(tai(nn,date_c%m)-gai(nn,date_c%m))*amnvv(nn)     ! land_snow
        ! tlai(nn) = gai(nn,date_c%m)
        ! tsai(nn) = sai(nn,date_c%m)
        ! tlai(nn) = MCD15A2H_LAI(ii-1,jj-1,nn,date_c%m)
        ! tsai(nn) = sai(nn,date_c%m)
        ! tlai(nn) = gai(nn,date_c%m)/maxval(gai(nn,:))*maxval(MCD15A2H_LAI(ii-1,jj-1,nn,:))
        ! tsai(nn) = sai(nn,date_c%m)
        
        ol = min( max(snow(ii,jj)-hbot(nn),0.), htop(nn)-hbot(nn))
        fb = 1. - ol / max(1.e-6, htop(nn)-hbot(nn))
        elai(nn) = tlai(nn)*fb
        esai(nn) = tsai(nn)*fb
        if (elai(nn) < 0.05) elai(nn) = 0.
        if (esai(nn) < 0.05) esai(nn) = 0.
        vai(nn) = elai(nn) + esai(nn)
        
    end subroutine

    ! внутренние процедуры
    ! -------------------------------------------------------------------------------------------------------------------

    subroutine albedotablecalc(nv, abs_table, fsun)
        ! ---------------------------------------
        use grid, only : ii, jj
        integer, intent(in) :: nv
        real, intent(out) :: abs_table(2,2)
        real, intent(out) :: fsun
        
        real :: albsoil_table(2,2)

        call albsoil_calc(albsoil_table)
        call albveg_calc(nv, abs_table, albsoil_table, cosz(ii,jj), fsun)

    end subroutine
    
    
    subroutine sunec()
        ! ---------------------------------------
        use const, only : pi, yrs, deg2rad
        use grid, only : i0, i1, j0, j1, lon, lat, date_c
    
        integer :: i, j

        ! coordinates of Sun in equatorial coordinate system
        real :: dec  ! declination, rad
        real :: ha   ! hour angle, rad

        real :: gst  ! Greenwich siderial time, rad; equals to 0 in midnight
        real :: tau  ! angle of Earth rotation around Sun; equals to 0 in January, 1st 00:00
        real :: teq  ! time equation, rad

        ! ZCLOCK=AMOD((FLOAT(NTBASE*NTIMST)+0.5E0*ASTEP*TWODT)/86400.0E0,1.0E0)*2.0E0*PI
        ! ZYTIME=AMOD((FLOAT(NCBASE-1)+(FLOAT(NTBASE*NTIMST)+0.5E0*ASTEP*TWODT)/86400.0E0)/365.2425E0,1.0E0)*2.0E0*PI
        gst = 2.*pi*(3600.*(date_c%h-date_c%UTC) + 60.*date_c%mn + date_c%sc)/86400.
        tau = 2.*pi*(24.*3600.*date_c%doy + 3600.*(date_c%h-date_c%UTC) + 60.*date_c%mn + date_c%sc)/(date_c%ndays*86400.)

        ! 0nd-order member in teq is 0.0000075 instead of 0.000075
        dec = 0.006918 - 0.399912*cos(tau) + 0.070257*sin(tau) - 0.006758*cos(2.*tau) + 0.000907*sin(2.*tau)
        teq = 0.000008 + 0.001868*cos(tau) - 0.032077*sin(tau) - 0.014615*cos(2.*tau) - 0.040849*sin(2.*tau)

        do i = i0, i1
            do j = j0, j1
                ha = gst + teq + lon(i)*deg2rad
                cosz(i,j) = sin(lat(j)*deg2rad)*sin(dec) + cos(lat(j)*deg2rad)*cos(dec)*cos(ha)
                if (cosz(i,j) < 0.) cosz(i,j) = 0.  ! Sun below horizont
            enddo
        enddo

    end subroutine

    
    subroutine albsoil_calc(albsoil_table)
        ! ---------------------------------------
        use environment_core, only : Wsoil
        use grid, only : ii, jj, ms
        real, intent(out) :: albsoil_table(2,2)
        real :: inc, albedo_dry, albedo_sat
        integer :: iband, num
        
        do iband = 1, 2
        
            inc = max(0.11 - 0.40 * Wsoil(ii,jj,ms+1)*1.2, 0.)
            albedo_dry = 0.
            albedo_sat = 0.
            do num = 1, 9
                albedo_dry = albedo_dry + albdry(num,iband) * soilcol(ii,jj,num)
                albedo_sat = albedo_sat + albsat(num,iband) * soilcol(ii,jj,num)
            end do
            albsoil_table(idir,iband) = min(albedo_sat+inc, albedo_dry)
            albsoil_table(idif,iband) = albsoil_table(idir,iband)
        enddo
        
    end subroutine
    
    
    subroutine albveg_calc (nv, abs_table, albsoil_table, cos_z,fsun_)
        ! ---------------------------------------
        use const, only : Kelvin0
        use environment_core, only : Tgr
        use grid, only : ii, jj
        
        integer, intent(in) :: nv
        real, intent(out) :: abs_table(2,2)  !< Flux abs by veg layer (per unit incoming flux)
        real, intent(in) :: albsoil_table(2,2)
        real, intent(in) :: cos_z
        real, intent(out) :: fsun_
        ! также elai, esai, fwet

        real, parameter :: tfrz = Kelvin0               !< Freezing temperature (kelvin)
        
        real rho(twoband,nv2)   !< Leaf+stem reflectance
        real tau(twoband,nv2)   !< Leaf+stem transmittance
        real :: gdir         !< Relative projected leaf+stem area in solar direction
        real omega   !fraction of intercepted radiation that is scatt      !ered
        real omegal  !omega for leaves
        real betai   !upscatter parameter for diffuse radiation
        real betail  !betai for leaves
        real betad   !upscatter parameter for direct beam radiation
        real betadl  !betad for leaves
        real ext     !optical depth of direct beam per unit leaf area
        real avmu    !average diffuse optical depth
        real cosz         !0.001 <= coszen <= 1.000
        real asu          !single scattering albedo
        real chil         ! -0.4 <= xl <= 0.6

        real tmp0,tmp1,tmp2,tmp3,tmp4,tmp5,tmp6,tmp7,tmp8,tmp9
        real p1,p2,p3,p4,s1,s2,u1,u2,u3
        real b,c,d,d1,d2,f,h,h1,h2,h3,h4,h5,h6,h7,h8,h9,h10
        real phi1,phi2,sigma
        real ftds,ftis,fres
        real :: tgr1
        
        integer ib              !< Waveband number
        integer ic              !< 0=unit incoming direct; 1=unit incoming diffuse
        
        real :: ext2
        real, parameter :: prevent = 0.000001
        real :: wl, ws

        tgr1 = Tgr(ii,jj) + 3.
        
        do ib = 1, 2
        do ic = 1, 2
        
            wl = elai(nv) / max(vai(nv),prevent)
            ws = esai(nv) / max(vai(nv),prevent)
            rho(ib,nv)=max(rhol(ib,nv)*wl+rhos(ib,nv)*ws, prevent)
            tau(ib,nv)=max(taul(ib,nv)*wl+taus(ib,nv)*ws, prevent)

            cosz = max(0.001, cos_z)
            chil = min( max(-0.4, xl(nv)), 0.6)
            if (abs(chil) <= 0.01) chil = 0.01
            phi1 = 0.5 - 0.633*chil - 0.330*chil*chil
            phi2 = 0.877 * (1.-2.*phi1)
            gdir = phi1 + phi2*cosz
            ext = gdir/cosz
            avmu = ( 1. - phi1/phi2 * log((phi1+phi2)/phi1) ) / phi2
            omegal = rho(ib,nv) + tau(ib,nv)
            tmp0 = gdir + phi2*cosz
            tmp1 = phi1*cosz
            asu = 0.5*omegal*gdir/tmp0 * ( 1. - tmp1/tmp0 * log((tmp1+tmp0)/tmp1) )
            betadl = (1.+avmu*ext)/(omegal*avmu*ext)*asu
            betail = 0.5* ( rho(ib,nv)+tau(ib,nv) + (rho(ib,nv)-tau(ib,nv)) * ((1.+chil)/2.)**2 ) / omegal
        
            if (tgr1 > tfrz) then
                omega = omegal
                betad = betadl
                betai = betail
            else
                omega =  (1.-fwet(nv))*omegal        + fwet(nv)*omegas(ib)
                betad = ((1.-fwet(nv))*omegal*betadl + fwet(nv)*omegas(ib)*betads) / omega
                betai = ((1.-fwet(nv))*omegal*betail + fwet(nv)*omegas(ib)*betais) / omega
            end if
        
            b = 1. - omega + omega*betai
            c = omega*betai
            tmp0 = avmu*ext
            d = tmp0 * omega*betad
            f = tmp0 * omega*(1.-betad)
            tmp1 = b*b - c*c
            h = sqrt(tmp1) / avmu
            sigma = tmp0*tmp0 - tmp1
            if(sigma == 0.) sigma = 0.0001
            p1 = b + avmu*h
            p2 = b - avmu*h
            p3 = b + tmp0
            p4 = b - tmp0
            s1 = exp(-h*vai(nv))
            s2 = exp(-ext*vai(nv))
            u1 = b - c/albsoil_table(ic,ib)!albsod(ib)
            u2 = b - c*albsoil_table(ic,ib)!albsod(ib)
            u3 = f + c*albsoil_table(ic,ib)!albsod(ib)
            tmp2 = u1 - avmu*h
            tmp3 = u1 + avmu*h
            d1 = p1*tmp2/s1 - p2*tmp3*s1
            tmp4 = u2 + avmu*h
            tmp5 = u2 - avmu*h
            d2 = tmp4/s1 - tmp5*s1
            h1 = -d*p4 - c*f
            tmp6 = d - h1*p3/sigma
            tmp7 = ( d - c - h1/sigma*(u1+tmp0) ) * s2
            h2 = ( tmp6*tmp2/s1 - p2*tmp7 ) / d1
            h3 = - ( tmp6*tmp3*s1 - p1*tmp7 ) / d1
            h4 = -f*p3 - c*d
            tmp8 = h4/sigma
            tmp9 = ( u3 - tmp8*(u2-tmp0) ) * s2
            h5 = - ( tmp8*tmp4/s1 + tmp9 ) / d2
            h6 = ( tmp8*tmp5*s1 + tmp9 ) / d2
            h7 = (c*tmp2) / (d1*s1)
            h8 = (-c*tmp3*s1) / d1
            h9 = tmp4 / (d2*s1)
            h10 = (-tmp5*s1) / d2

            select case (ic)
                case(idir)
                    fres = h1/sigma + h2 + h3
                    ftds = s2
                    ftis = h4*s2/sigma + h5*s1 + h6/s1
                case(idif)
                    fres = h7 + h8          ! flux reflected by vegetation
                    ftds = 0.
                    ftis = h9*s1 + h10/s1   ! downward direct and diffuse fluxes below vegetation
            end select
            
            abs_table(ic,ib) = 1. - fres - (1.-albsoil_table(idir,ib))*ftds - (1.-albsoil_table(idif,ib))*ftis                  ! flux absorbed by vegetation

            enddo
            enddo
            
        ! fsun
        ! здесь другой prevent и ext
        ext2 = gdir/max(cos_z,prevent) * sqrt(1.-rho(1,nv)-tau(1,nv))
        fsun(nv) = (1.-exp(-ext2*vai(nv))) / max(ext2*vai(nv),prevent)
        if (fsun(nv) < 0.01) fsun(nv) = 0.
        
        fsha(nv) = 1.-fsun(nv)
        
        fsun_ = fsun(nv)
        
    end subroutine albveg_calc



    subroutine special_soilpar_read(filename, varname, var, fill_v)
        ! ---------------------------------------
        use grid, only : nlon, nlat, ich
        use netcdf
        use netcdf_kit, only : nc_errhand
    
        character(*), intent(in) :: filename
        character(*), intent(in) :: varname
        real, intent(out) :: var(:,:)
        real, intent(in) :: fill_v
    
        real, allocatable :: work2r(:,:)
        real :: fill_v_1, fill_v_2
        integer :: ncid, varid
        integer :: i, j

        allocate(work2r(nlon,nlat))
		print*, filename
        call nc_errhand( nf90_open(filename, nf90_nowrite, ncid) )
        call nc_errhand( nf90_inq_varid(ncid, varname, varid) )
        call nc_errhand( nf90_get_att(ncid, varid, 'fill_value', fill_v_1) )
        call nc_errhand( nf90_get_var(ncid, varid, work2r(nlon/2+1:nlon,:), (/1,1/), (/nlon/2,nlat/)) )
        call nc_errhand( nf90_get_var(ncid, varid, work2r(1:nlon/2,:), (/nlon/2+1,1/), (/nlon/2,nlat/)) )
        fill_v_2 = maxval(work2r)
        do i = 1, nlon
            do j = 1, nlat
                if (ich(i,j) > 0 .and. work2r(i,j) /= fill_v_1 .and. work2r(i,j) /= fill_v_2) then
                    var(i,j) = work2r(i,j)
                else
                    var(i,j) = fill_v
                endif
            enddo
        enddo
        call nc_errhand( nf90_close(ncid) )
        deallocate(work2r)
        
    end subroutine
    
    
end module
