module environment_data_station

    ! интерфейс
! ----------------------------------------------- Use pack ------------------------------------------------------------
    use environment_core!, only: Temp, e, Rlwd, Rswd, p, pr, Wind, ra, Tsrf, Tsoil, Wsoil, Isoil, snow, Tgr, &
    !& sw, bettar, kd, veg, lambd, organic, FYM, rhodry, g2gw_to_cmw
    use grid,   only: date_c, date_fst, date_lst, dt, i0, i1, j0, j1, ml, ms,dz
	use config, only: station_name, station_opt
	use const,  only: yrs, nmonth, pi
! ---------------------------------------------- Variables ------------------------------------------------------------
    implicit none

    private
    public :: environment_data_init
    public :: environment_data_calc_at_timestep
    public :: environment_data_calc_at_cell
    public :: environment_data_calc_at_tile
! ---------------------------------------------------------------------------------------------------------------------
    ! ------- Serve value -------
    integer :: mnc                         !< Количество месяцев в расчете
    integer :: mncX = 0                    !< Номер месяца с начала работы программы
	integer, parameter :: station_max = 5  !< Максимальное число станций
	integer, parameter :: opt_max = 4      !< Максимальное количество способов подачи удобрения    
    integer :: opt_n                       !< Номер способа подачи удобрения
	integer :: station_n !                 DAO4 DAO3 TRGK VLDR ROST   !< Номер станции наблюдения за климатом
                         !                  1    2    3    4    5
    integer :: mnclot_fst(station_max) = (/1935,1937,1956,1968,1975/) !< Даты начала сбора данных по климату, в рамках станции 
    integer :: mnclot_lst(station_max) = (/2012,2012,2018,2018,2018/) !< Даты конца сбора данных по климату, в рамках станции 
! ---------------------------------------------------------------------------------------------------------------------
    ! ------ Determined externally  ------
	real(kind = 16), allocatable :: in_temp (:)    !< Поступление извне данных cредней температуры воздуха в месяц, [Celsius]
	real(kind = 16), allocatable :: in_veg  (:)    !< Поступление извне данных средней концентрации растений в месяц, [%]
	real(kind = 16), allocatable :: in_wsoil(:)    !< Поступление извне данных влажности почвы, [dim]
    
    real(kind = 16), allocatable :: in_organic(:) !< Поступление извне данных по поступлению органического углерода в почву, [kg/m**3 / year]
    real(kind = 16), allocatable :: in_FYM(:)     !< Поступление извне данных по поступлению навова в почву, [kg/m**3 / year]
    
    real, allocatable :: in_rainfall(:)            !< Поступление извне данных влажности почвы, [mm]
    real, allocatable :: in_Evpar(:)               !< Поступление извне данных влажности почвы, [mm]

    !                          station        =   DAO4      DAO3      TRGK      VLDR     ROST
    ! ---- For INMCM ----
    real(kind = 16) :: rhodry_in(station_max) = (/1.39,     0.8,     1.4,     1.35,    2.2/)
	                                                       !1.29                        ! 1.15
    ! ---- For Rothc ----
    real(kind = 16) :: sw_in    (station_max) = (/0.120,    0.120,    0.120,    0.120,   0.120/) !0.146
    real(kind = 16) :: bettar_in(station_max) = (/20.,      26.,      7.,       8.,      44./)

    ! ---- For SOCs ----
    real :: kd_in    (station_max,opt_max)	
	!          opt =    1      2      3      4
    data kd_in(01,:)   /0.03,  0.03,  0.035, 0.03/
    data kd_in(02,:)   /0.03,  0.03,  0.035, 0.03/  !0.08581933
    data kd_in(03,:)   /0.0402, 0.05067, 0.04185, 0./ !0.015,    0.008,    0.008,    0.
    data kd_in(04,:)   /0.08581933, 0.08581933, 0.08581933, 0./ !0.015,    0.008,    0.008,    0.
    data kd_in(05,:)   /0.012, 0.007, 0.007, 0./

contains

    ! внешние процедуры
    ! ---------------------------------------------------------------------------------

    subroutine environment_data_init()

        use const, only : miss_v, rho_w

        integer :: k                    !<count

        mnc = (date_lst%y - date_fst%y)*nmonth + (date_lst%m - date_fst%m)
        allocate(in_organic(0:mnc))
        allocate(in_FYM(0:mnc))

        allocate(in_temp (0:mnc))
        allocate(in_veg  (0:mnc))
        allocate(in_wsoil(0:mnc))
        allocate(in_Evpar(0:mnc))
        allocate(in_rainfall(0:mnc))

        kd_in = kd_in/yrs 

        read(station_opt,*) opt_n

        select case (station_name)
          case('DAO4')			
            station_n = 1
          case('DAO3')
            station_n = 2
          case('TRGK')
            station_n = 3
          case('VLDMR')
            station_n = 4
          case('Rostov')
            station_n = 5
        end select		

       ! External carbon intake
        open (unit = 1, file = 'initial_value/'//'carbon_intake/'//trim(station_name)//'_org_'//trim(station_opt)//'.txt', status='unknown')
        read(1,*) in_organic(1:mnc)
        close (1)

        open (unit = 1, file = 'initial_value/'//'carbon_intake/'//trim(station_name)//'_FYM_'//trim(station_opt)//'.txt', status='unknown')
        read(1,*) in_FYM(1:mnc)
        close (1)

       ! External climate intake
        open (unit = 1, file = 'initial_value/'//'climate_intake/'//trim(station_name)//'_Temp.txt', status='unknown')
        read(1,*) in_temp(1:mnc)
        close (1)

        open (unit = 1, file = 'initial_value/'//'climate_intake/'//trim(station_name)//'_MOI.txt', status='unknown')
        read(1,*) in_wsoil(1:mnc)
        close (1)

        open (unit = 1, file = 'initial_value/'//'climate_intake/'//trim(station_name)//'_Veg.txt', status='unknown')
        read(1,*) in_veg(1:mnc)
        close (1)

        open (unit = 1, file = 'initial_value/'//'climate_intake/'//trim(station_name)//'_Rainfall.txt', status='unknown')
        read(1,*) in_rainfall(1:mnc)
        close (1)

        open (unit = 1, file = 'initial_value/'//'climate_intake/'//trim(station_name)//'_Evpar.txt', status='unknown')
        read(1,*) in_Evpar(1:mnc)
        close (1)

        in_organic = in_organic/dt
        in_FYM = in_FYM/dt

        rhodry(:,:,:) = rhodry_in(station_n)
        sw    (:,:)   = sw_in    (station_n)
        bettar(:,:)   = bettar_in(station_n)
        kd    (:,:)   = kd_in    (station_n, opt_n)

        do k = ms+1, ml-1
            g2gw_to_cmw(:,:,k) = rhodry(:,:,k) / rho_w * dz(k)
        end do

! ---- Part of timesteps init ----
! @todo учесть шаг по времени 1 час
      if (dt == 86400) then ! Для шага по времени день:
        mncX = (date_fst%y - mnclot_fst(station_n))*nmonth + date_lst%m - date_fst%m
        in_temp (0) = in_temp (1)
        in_veg  (0) = in_veg  (1)
        in_wsoil(0) = in_wsoil(1)
        in_organic(0) = in_organic(1)
        in_FYM(0)   = in_FYM(1)
        in_Evpar(0) = in_Evpar(1)
        in_rainfall(0) = in_rainfall(1)
      else                  ! Для шага по времени месяц:
        mncX = (date_fst%y - mnclot_fst(station_n))*nmonth + date_lst%m
      end if
	  
! Набор переменных которые должны быть, но не данны через качественные внешние данные
      p(:,:) = 980.                                  !< pressure
      Rlwd(:,:) = 350.                               !< longwave radiation
      ra(:,:) = 20.                                  !< aerodynamical resistance
      snow(:,:) = 0.                                 !< snow cover 
      Isoil(:,:,:) = 0.                              !< part of ice in the soil	  

    end subroutine


    subroutine environment_data_calc_at_timestep()
        ! ---------------------------------------
        use const, only : w_day, miss_v
        use grid,  only : z, ms, ml
		
        integer nom,nod, N(12)
        integer :: i, j       !< count
        real :: hour
    ! ------------- Переменные для распределения значений по всему временному интервалу
        nom = date_c%m             ! Номер месяца
        nod = date_c%d             ! Номер дня месяца
        N(nom) = date_c%days(nom)  ! Количество дней в месяце
        !mncX                      ! Номер месяца с начала работы программы
    ! -------------
      if (dt == 86400) then ! Для шага по времени день:
        organic(:,:) = in_organic(mncX)/N(nom)
        FYM    (:,:) = in_FYM(mncX)/N(nom)
        Temp   (:,:) =  in_temp(mncX)*sin(pi*(nod-1)/(N(nom)-1)) +   in_temp(mncX+1)*(1-sin(pi*(nod-1)/(N(nom)-1)))
        Wsoil(:,:,:) = in_wsoil(mncX)*sin(pi*(nod-1)/(N(nom)-1)) +  in_wsoil(mncX+1)*(1-sin(pi*(nod-1)/(N(nom)-1)))
        veg    (:,:) =   in_veg(mncX)
        Evpar  (:,:) = in_Evpar(mncX)/N(nom)
        pr     (:,:) = in_rainfall(mncX)/N(nom)
        if (nod == (N(nom)/2)) mncX = mncX + 1
      else                  ! Для шага по времени месяц:
        organic(:,:) = in_organic(mncX)
        FYM    (:,:) = in_FYM(mncX)
        Temp   (:,:) =  in_temp(mncX)
        Wsoil(:,:,:) = in_wsoil(mncX)
        veg    (:,:) =   in_veg(mncX)
        Evpar  (:,:) = in_Evpar(mncX)
        pr     (:,:) = in_rainfall(mncX)
        mncX = mncX + 1
      end if

      lambd(:,:) = organic(:,:) + FYM(:,:)

      !if (mncX == mnc) mncX = 0 !для инициализации, по идее, в других случаях строка комментируется

	  do i = 1, ml
        Tsoil(:,:,i) = Temp(:,:)     ! температура почвы равна температуре воздуха
      end do

	  hour = date_c%h
! Набор переменных которые должны быть, но не данны через качественные внешние данные
      do i = i0, i1
          do j = j0, j1
              Wind(i,j) = 2 + 1.*cos(w_day*(hour-13))        !< wind
              e(i,j) = 15. + 2.*cos(2*w_day*(hour-7))        !< humidity
              Rswd(i,j) = max(0.,800.*cos(w_day*(hour-12)))  !< shortwave radiation
              ! ---
              Tsrf(i,j) = 20. + 8.*cos(w_day*(hour-12.5))    !< temperature of surface
              Tgr(i,j) = Tsrf(i,j)
          end do
      end do

    end subroutine

    
    subroutine environment_data_calc_at_cell(ii,jj)
		
        integer, intent(in) :: ii, jj

    end subroutine
    
    subroutine environment_data_calc_at_tile(ii,jj,nn)
        ! ---------------------------------------
        integer, intent(in) :: ii, jj, nn
    end subroutine

end module
