
module datetime

    ! интерфейс
    ! ---------------------------------------------------------------------------------

    use const, only : miss_v

    implicit none

    private
    public :: date_type
    
    ! параметры
    ! ---------------------------------------------------------------------------------
    
    integer, parameter :: len_timestamp = 19
    character(*), parameter :: fmt_timestamp_write = '(i4.4,"-",i2.2,"-",i2.2," ",i2.2,":",i2.2,":",i2.2)'  ! формат меток [yyyy-mm-dd hh:mm:ss]
    character(*), parameter :: fmt_timestamp_read =  '(i4.4, 1x,i2.2, 1x,i2.2, 1x,i2.2, 1x,i2.2, 1x,i2.2)'
    real(8), parameter :: jday_ref = 693594.  ! референсное значение на дату 1899-12-30 00:00:00' (как в Excel)

    integer, parameter :: ndays_400y = 303*365 + 97*366
    integer, parameter :: ndays_100y = 76*365 + 24*366
    integer, parameter :: ndays_4y = 3*365 + 366
    integer, parameter :: ndays_1y = 365
    
    ! производные типы
    ! ---------------------------------------------------------------------------------
    
    type date_type
        integer :: y
        integer :: m
        integer :: d
        integer :: h
        integer :: mn
        integer :: sc
        real(8) :: jday                        ! абсолютное время (юлианский день)
        character(len_timestamp) :: timestamp  ! текстовая метка
        integer :: UTC                         ! часовой пояс
        ! диганостика:
        integer :: ndays
        integer :: days(12)
        integer :: doy
        integer :: dec
        contains
            procedure, pass :: init => date_init
            procedure, pass :: shift => date_shift
    end type date_type

    
contains
    
    
    ! внутренние процедуры
    ! ---------------------------------------------------------------------------------
    
    subroutine date_init(date, timestamp, jday, UTC)
        ! ---------------------------------------
        class(date_type), intent(inout) :: date
        character(len_timestamp), optional, intent(in) :: timestamp
        real(8), optional, intent(in) :: jday
        integer, optional, intent(in) :: UTC
        
        if (present(timestamp) .and. present(jday)) then
            stop "check failed : timestamp and jday at the same time at date_init"
        elseif (present(timestamp)) then
            call timestamp_to_date(timestamp, date)
        elseif (present(jday)) then
            call jday_to_date(jday, date)
        else
            stop "check failed : no arguments at date_init"
        endif
        
        if (present(UTC)) then
            date%UTC = UTC
        else
            date%UTC = 0
        endif
        
        call date_diagnostics(date)
        
    end subroutine


    subroutine date_shift(date, dt)
        ! ---------------------------------------
        !< @details оптимально только на небольших шагах по времени

        class(date_type), intent(inout), target :: date
        integer, intent(in) :: dt  ! в секундах, больше нуля
        integer, pointer :: y, m, d, h, mn, sc
        integer :: days(12)

        if (dt < 0.) stop "check failed : date_shift requires dt > 0"
        
        y => date%y
        m => date%m
        d => date%d
        h => date%h
        mn => date%mn
        sc => date%sc
        
        days = days_in_year(y)
        
        sc = sc + dt
        if (sc >= 60) then
            mn = mn + sc/60
            sc = mod(sc,60)
            if (mn >= 60) then
                h = h + mn/60
                mn = mod(mn,60)
                if (h >= 24) then
                    d = d + h/24
                    h = mod(h,24)
                    do while (d > days(m))
                        d = d - days(m)
                        m = m + 1
                        if (m > 12) then
                            m = 1
                            y = y + 1
                            days = days_in_year(y)
                        end if
                    end do
                end if
            end if
        end if

        call date_diagnostics(date)
        
    end subroutine date_shift



    subroutine date_diagnostics(date)
        ! ---------------------------------------
        type(date_type), intent(inout) :: date
        integer :: i, work
        
        call date_to_jday(date, date%jday)
        call date_to_timestamp(date, date%timestamp)
        
        date%days = days_in_year(date%y)
        date%ndays = sum(date%days)
        
        work = 0
        do i = 1, date%m-1
            work = work + date%days(i)
        enddo
        work = work + date%d
        date%doy = work
        
        if (1 <= date%d .and. date%d <= 10) then
            date%dec = 1
        elseif (11 <= date%d .and. date%d <= 20) then
            date%dec = 2
        elseif (21 <= date%d .and. date%d <= 31) then
            date%dec = 3
        else
            date%dec = miss_v
        endif
        
    end subroutine



    function days_in_year(y)
        ! ---------------------------------------
        !< @brief function days_in_year(y) counts leap years and days in a month
        
        integer, intent(in) :: y

        logical :: is_leap
        integer, parameter :: days_norm(12) = (/31,28,31,30,31,30,31,31,30,31,30,31/)
        integer, parameter :: days_leap(12) = (/31,29,31,30,31,30,31,31,30,31,30,31/)
        integer :: days_in_year(12)

        is_leap = (mod(y,4) == 0. .and. (mod(y,100) /= 0 .or. mod(y,400) == 0))
        if (is_leap) then
            days_in_year = days_leap
        else
            days_in_year = days_norm
        end if

    end function days_in_year

        
    subroutine date_to_timestamp(date, timestamp)
        ! ---------------------------------------
        type(date_type), intent(in) :: date
        character(len_timestamp), intent(out) :: timestamp

        write(timestamp, fmt_timestamp_write) date%y, date%m, date%d, date%h, date%mn, date%sc
    
    end subroutine
    
    
    subroutine timestamp_to_date(timestamp, date)
        ! ---------------------------------------
        character(len_timestamp), intent(in) :: timestamp
        type(date_type), intent(out) :: date
        
        read(timestamp, fmt_timestamp_read) date%y, date%m, date%d, date%h, date%mn, date%sc

    end subroutine
    
    
    subroutine date_to_jday(date, jday)
        ! ---------------------------------------
        type(date_type), intent(in), target :: date
        real(8), intent(out) :: jday
        
        integer, pointer :: y, m, d, h, mn, sc
        integer :: work, n400y, n100y, n4y, n1y, days(12), doy, i
        real(8) :: jday_abs
        
        y => date%y
        m => date%m
        d => date%d
        h => date%h
        mn => date%mn
        sc => date%sc
        
        work = y - 1
        n400y = work / 400
        work = mod(work,400)
        n100y = work / 100
        work = mod(work,100)
        n4y = work / 4
        work = mod(work,4)
        n1y = work
        
        days = days_in_year(y)
        doy = 0
        do i = 1, m - 1
            doy = doy + days(i)
        enddo
        doy = doy + d
        
        jday_abs = n400y*ndays_400y + n100y*ndays_100y + n4y*ndays_4y + n1y*ndays_1y + doy
        jday_abs = jday_abs + (h + mn/60. + sc/3600.)/24.
        jday = jday_abs - jday_ref
        
    end subroutine
    
    
    subroutine jday_to_date(jday, date)
        ! ---------------------------------------
        real(8), intent(in) :: jday
        type(date_type), intent(out), target :: date
        
        integer, pointer :: y, m, d, h, mn, sc
        integer :: work, n400y, n100y, n4y, n1y, days(12)
        real(8) :: jday_abs
        
        y => date%y
        m => date%m
        d => date%d
        h => date%h
        mn => date%mn
        sc => date%sc
        
        jday_abs = jday + jday_ref
        
        work = floor(jday_abs)
        n400y = work / ndays_400y
        work = mod(work,ndays_400y)
        n100y = work / ndays_100y
        work = mod(work,ndays_100y)
        n4y = work / ndays_4y
        work = mod(work,ndays_4y)
        n1y = work / ndays_1y
        work = mod(work,ndays_1y)
        y = 400*n400y + 100*n100y + 4*n4y + n1y + 1
        
        days = days_in_year(y)
        m = 1
        do while (work > days(m))
            work = work - days(m)
            m = m + 1
        enddo
        d = work
        
        work = nint(86400*(jday_abs-int(jday_abs)))
        h = work / 3600
        work = mod(work,3600)
        mn = work / 60
        work = mod(work,60)
        sc = work
        
    end subroutine
        
end module
