
module carbon_postprocessing

    ! интерфейс
    ! ---------------------------------------------------------------------------------

    use carbon_core, only : pool_type, flux_type, pool, flux, npool_default, nflux_default, npool, nflux

    implicit none
    
    private
    public :: carbon_standard_print
    public :: carbon_standard_output
    public :: carbon_testing_write_log
    public :: carbon_testing_read_log
    
    ! переменные
    ! ---------------------------------------------------------------------------------
    
    integer :: ncid, t_id_v
    integer, allocatable :: pool_id(:), flux_id(:)
    
    type(pool_type) :: pool_ref(npool_default), eps_pool(npool_default)
    type(flux_type) :: flux_ref(nflux_default), eps_flux(nflux_default)

    type(pool_type), target :: pool_sum_year(npool_default)
    type(flux_type), target :: flux_sum_year(nflux_default)
    integer :: n_sum_year
    
    
    
    
contains


    ! внешние процедуры
    ! ---------------------------------------------------------------------------------
    
    subroutine carbon_testing_write_log()
        ! ---------------------------------------
        use const, only : miss_v, len_default
        use paths, only : path_logs
        use grid, only : i0, i1, j0, j1, tt, ni, nj
        use carbon_core, only : ntile
        use config, only : carbon_model_type
        
        integer :: i, j, n, k
        integer :: rec_len
        character(len_default) :: logname
        logical, save :: firstcall = .true.
        
        if (npool + nflux > 0) then
            if (firstcall) then
                do n = 1, npool
                    allocate(eps_pool(n)%val, source = pool(n)%val)
                    eps_pool(n)%val(:,:,:) = miss_v
                enddo
                do n = 1, nflux
                    allocate(eps_flux(n)%val, source = flux(n)%val)
                    eps_flux(n)%val(:,:,:) = miss_v
                enddo
                rec_len = (npool+nflux)*(ni*nj*ntile)*4
                logname = path_logs//'error_test_log_'//trim(carbon_model_type)//'.bin'
                open(100, file=trim(logname), status='replace',  access='direct', recl=rec_len)
                firstcall = .false.
            endif
            write(100,rec=tt) ((((pool(k)%val(i,j,n),i=i0,i1),j=j0,j1),n=1,ntile),k=1,npool), &
                            & ((((flux(k)%val(i,j,n),i=i0,i1),j=j0,j1),n=1,ntile),k=1,nflux)
        endif
        
    end subroutine


    subroutine carbon_testing_read_log()
        ! ---------------------------------------
        !< @brief отслеживание технических ошибок
        
        use const, only : miss_v, len_default
        use paths, only : path_logs
        use config, only : carbon_model_type
        use carbon_core, only : ntile
        use grid, only : i0, i1, j0, j1, tt, ni, nj
        
        integer :: i, j, n, k
        integer :: rec_len, ios
        character(len_default) :: logname
        logical, save :: firstcall = .true.
        logical, save :: ex
        
        if (firstcall) then
            do n = 1, npool
                allocate(pool_ref(n)%val, source = pool(n)%val)
                allocate(eps_pool(n)%val, source = pool(n)%val)
                eps_pool(n)%val(:,:,:) = miss_v
            enddo
            do n = 1, nflux
                allocate(flux_ref(n)%val, source = flux(n)%val)
                allocate(eps_flux(n)%val, source = flux(n)%val)
                eps_flux(n)%val(:,:,:) = miss_v
            enddo
            logname = path_logs//'error_test_log_'//trim(carbon_model_type)//'.bin'
            inquire(file=trim(logname), exist = ex)
            if (ex) then
                rec_len = (npool+nflux)*(ni*nj*ntile)*4
                open(100, file=trim(logname), status='old', access='direct', recl=rec_len)
            endif
            firstcall = .false.
        endif
        
        if (ex) then
            read(100,rec=tt,iostat=ios) ((((pool_ref(k)%val(i,j,n),i=i0,i1),j=j0,j1),n=1,ntile),k=1,npool), &
                                      & ((((flux_ref(k)%val(i,j,n),i=i0,i1),j=j0,j1),n=1,ntile),k=1,nflux)
            if (ios == 0) then
                do n = 1, npool
                    eps_pool(n)%val(:,:,:) = pool(n)%val(:,:,:) - pool_ref(n)%val(:,:,:)
                enddo
                do n = 1, nflux
                    eps_flux(n)%val(:,:,:) = flux(n)%val(:,:,:) - flux_ref(n)%val(:,:,:)
                enddo
            else
                do n = 1, npool
                    eps_pool(n)%val(:,:,:) = miss_v
                enddo
                do n = 1, nflux
                    eps_flux(n)%val(:,:,:) = miss_v
                enddo
            endif
        endif
        
    end subroutine
    

    subroutine carbon_standard_print()
        ! ---------------------------------------
        !< @brief стандартный вывод данных в консоль

        use config, only : nv_singlecolumn, testing_log_mode
        use const, only : umol2kg, miss_v
        use grid, only : i0, j0, date_c, tt
        use carbon_core, only : ntile

        integer :: n, n0, n1
        
        if (nv_singlecolumn /= miss_v) then
            n0 = nv_singlecolumn
            n1 = nv_singlecolumn
        else
            n0 = 1
            n1 = ntile
        endif
        
        print'(1x,a,1x,i10)', date_c%timestamp, tt
        
        print*, 'пулы (кг/м2):'
        select case(testing_log_mode)
            case('none')
                do n = 1, npool
                    print*, n, pool(n)%val(i0,j0,n0:n1), trim(pool(n)%name)
                enddo
            case('write','read')
                if (tt == 0) then
                    do n = 1, npool
                        print*, n, pool(n)%val(i0,j0,n0:n1), miss_v, trim(pool(n)%name)
                    enddo
                else
                    do n = 1, npool
                        print*, n, pool(n)%val(i0,j0,n0:n1), eps_pool(n)%val(i0,j0,n0:n1), trim(pool(n)%name)
                    enddo
                endif
        end select
        
        print*, 'потоки (мкмоль/м2/с):'
        select case(testing_log_mode)
            case('none')
                do n = 1, nflux
                    print*, n, flux(n)%val(i0,j0,n0:n1), trim(flux(n)%name)
                enddo
            case('write','read')
                if (tt == 0) then
                    do n = 1, nflux
                        print*, n, flux(n)%val(i0,j0,n0:n1), miss_v, trim(flux(n)%name)
                    enddo
                else
                    do n = 1, nflux
                        print*, n, flux(n)%val(i0,j0,n0:n1), eps_flux(n)%val(i0,j0,n0:n1), trim(flux(n)%name)
                    enddo
                endif
        end select
        
        print*
        
    end subroutine
    
    
    subroutine carbon_standard_output()
        ! ---------------------------------------
        !< @brief стандартный вывод данных в файлы

        use config, only : if_out_yearly
        use const, only : umol2kg, miss_v
        use grid, only : i0, j0, date_c
        use datetime, only : date_type
        use paths, only : path_out
        logical, save :: firstcall = .true.
        integer :: n
        
        type(date_type), save :: date_mem
        data date_mem%h /miss_v/
        data date_mem%d /miss_v/
        data date_mem%m /miss_v/
        data date_mem%d /miss_v/
        
        logical :: if_hourly_output
        logical :: if_daily_output
        logical :: if_monthly_output
        logical :: if_yearly_output
        
        if (date_c%h /= date_mem%h) then
            if_hourly_output = .true.
            date_mem%h = date_c%h
        else
            if_hourly_output = .false.
        end if

        !> ежедневный
        if (date_c%d /= date_mem%d) then
            if_daily_output = .true.
            date_mem%d = date_c%d
        else
            if_daily_output = .false.
        end if

        !> ежемесячный
        if (date_c%m /= date_mem%m) then
            if_monthly_output = .true.
            date_mem%m = date_c%m
        else
            if_monthly_output = .false.
        end if

        !> ежегодный
        if (if_out_yearly .and. date_c%y /= date_mem%y) then
            if_yearly_output = .true.
            date_mem%y = date_c%y
        else
            if_yearly_output = .false.
        end if
        
        
        if (if_yearly_output) then
            
            if (firstcall) then
                call nc_create(path_out//'out_year.nc', ncid)
                firstcall = .false.
            else
                call nc_write(ncid)
                do n = 1, npool
                    pool_sum_year(n)%val(:,:,:) = 0.
                enddo
                do n = 1, nflux
                    flux_sum_year(n)%val(:,:,:) = 0.
                enddo
                n_sum_year = 0
            endif
            
            do n = 1, npool
                pool_sum_year(n)%val(:,:,:) = pool_sum_year(n)%val(:,:,:) + pool(n)%val(:,:,:)
            enddo
            do n = 1, nflux
                flux_sum_year(n)%val(:,:,:) = flux_sum_year(n)%val(:,:,:) + flux(n)%val(:,:,:)
            enddo
            n_sum_year = n_sum_year + 1
            
        endif
            
    end subroutine
    
    ! внутренние процедуры
    ! ---------------------------------------------------------------------------------
    
    subroutine nc_create(filename, ncid)
        ! ---------------------------------------
        use netcdf
        use netcdf_kit, only : nc_errhand
        use grid, only : i0, i1, j0, j1, lon, lat, ni, nj
        use const, only : deg2rad
    
        character(*), intent(in) :: filename
        integer, intent(out) :: ncid
        integer :: n, i_id, j_id, t_id, lon_id, lat_id
        
        call nc_errhand( nf90_create(filename, nf90_netcdf4, ncid) )
        call nc_errhand( nf90_def_dim(ncid, 'lon', i1-i0+1, i_id) )
        call nc_errhand( nf90_def_dim(ncid, 'lat', j1-j0+1, j_id) )
        call nc_errhand( nf90_def_dim(ncid, 't', nf90_unlimited, t_id) )
        call nc_errhand( nf90_def_var(ncid, 'lon', nf90_float, i_id, lon_id) )
        call nc_errhand( nf90_def_var(ncid, 'lat', nf90_float, j_id, lat_id) )
        call nc_errhand( nf90_def_var(ncid, 'time', nf90_int, t_id, t_id_v) )
        allocate(pool_id(npool))
        allocate(flux_id(nflux))
        do n = 1, npool
            call nc_errhand( nf90_def_var(ncid, trim(pool(n)%name), nf90_float, (/i_id, j_id, t_id/), pool_id(n)) )
        enddo
        do n = 1, nflux
            call nc_errhand( nf90_def_var(ncid, trim(flux(n)%name), nf90_float, (/i_id, j_id, t_id/), flux_id(n)) )
        enddo
        call nc_errhand( nf90_enddef(ncid) )
        
        call nc_errhand( nf90_put_var(ncid, lon_id, 0.01*nint(lon(:)*100), (/1/), (/ni/)) )
        call nc_errhand( nf90_put_var(ncid, lat_id, 0.01*nint(lat(:)*100), (/1/), (/nj/)) )
        
        do n = 1, npool
            allocate(pool_sum_year(n)%val, source = pool(n)%val)
        enddo
        do n = 1, nflux
            allocate(flux_sum_year(n)%val, source = flux(n)%val)
        enddo
        
    end subroutine
    
    
    subroutine nc_write(ncid)
        ! ---------------------------------------
        use netcdf
        use netcdf_kit, only : nc_errhand
        use grid, only : i0, i1, j0, j1, ni, nj
        use carbon_core, only : ntile, tile_weight
        use const, only : umol2kg
    
        integer, intent(in) :: ncid
        real, allocatable :: work(:,:)
        integer :: i, j, n, nn
        integer, save :: t = 0
        
        t = t + 1

        call nc_errhand( nf90_put_var(ncid, t_id_v, (/t/), (/t/), (/1/)) )
        
        allocate(work(i0:i1,j0:j1))
        do n = 1, npool
            work(:,:) = 0.
            do i = i0, i1
                do j = j0, j1
                    do nn = 1, ntile
                        if (nn >= 1 .and. nn <= 5) then
                            work(i,j) = work(i,j) + pool_sum_year(n)%val(i,j,nn) * tile_weight(i,j,nn)
                        endif
                    enddo
                    work(i,j) = work(i,j) / n_sum_year
                enddo
            enddo
            call nc_errhand( nf90_put_var(ncid, pool_id(n), work(:,:), (/1,1,t/), (/ni,nj,1/)) )
        enddo
        
        do n = 1, nflux
            work(:,:) = 0.
            do i = i0, i1
                do j = j0, j1
                    do nn = 1, ntile
                        work(i,j) = work(i,j) + flux_sum_year(n)%val(i,j,nn) * tile_weight(i,j,nn)
                    enddo
                    work(i,j) = work(i,j) / umol2kg
                enddo
            enddo
            call nc_errhand( nf90_put_var(ncid, flux_id(n), work(:,:), (/1,1,t/), (/ni,nj,1/)) )
        enddo
        
    end subroutine




end module
