module carbon_model_rothc_aux

    ! интерфейс
! ----------------------------------------------- Use pack ------------------------------------------------------------
 	use environment_core, only: Tsoil, Wsoil, organic, FYM, veg, sw, bettar, Evpar, pr
	use config, only: station_name, station_opt, carbon_model_type
    use const,  only: yrs, pi, nmonth
    use grid,   only: date_c, i0, i1, j0, j1, ml, dt
! ---------------------------------------------- Variables ------------------------------------------------------------
    implicit none

    ! ------- Pools -------
    real, dimension(:,:,:), pointer :: Catm            !< атмосфера
    real, dimension(:,:,:), pointer :: Cveg            !< растительность
    real, dimension(:,:,:), pointer :: CDPM            !< разлагаемый растительный материал / decomposable plant material
    real, dimension(:,:,:), pointer :: CRPM            !< устойчивый растительный материал / resistant plant material
    real, dimension(:,:,:), pointer :: CBIO            !< пул микробной биомассы / microbial biomass pool
    real, dimension(:,:,:), pointer :: CHUM            !< долгоживущий гумусовый пул / long lived humified pool
    real, dimension(:,:,:), pointer :: CIOM            !< инертное органическое вещество / inert organic matter
    ! ------- Flows -------
    real, dimension(:,:,:), pointer :: litterfall1     !< Между растениями и почвой DPM
    real, dimension(:,:,:), pointer :: litterfall2     !< Между растениеми и почвой RPM
    real, dimension(:,:,:), pointer :: atmosphere1     !< Между атмосферой и почвой BIO
    real, dimension(:,:,:), pointer :: atmosphere2     !< Между атмосферой и почвой HUM
    real, dimension(:,:,:), pointer :: respiration1    !< Между почвой DPM и атмосферой    ! 1 - DPM
    real, dimension(:,:,:), pointer :: respiration2    !< Между почвой RPM и атмосферой    ! 2 - RPM
    real, dimension(:,:,:), pointer :: respiration3    !< Между почвой BIO и атмосферой    ! 3 - BIO
    real, dimension(:,:,:), pointer :: respiration4    !< Между почвой HUM и атмосферой    ! 4 - HUM
    ! ------- Serve value -------
    integer, parameter :: ntiles = 4                   !< В данном случае, номер пула
    ! ------- Initial value ------
    real :: Rs = 0.0                                   !< Функции дыхания
    real :: RDPM = 0.0                                 !< Дыхание пула DPM
    real :: RRPM = 0.0                                 !< Дыхание пула RPM
    real :: RBIO = 0.0                                 !< Дыхание пула BIO
    real :: RHUM = 0.0                                 !< Дыхание пула HUM
    ! ------- Functions ------------
    real :: Ft = 0.                                    !< Функция температуры, [dim]
    real :: Fs = 0.                                    !< Функция влажности, [dim]
    real :: Fv = 0.                                    !< Функция фракции растительного покрова, [dim]           
    real :: fdpm = 0.                                  !< Функция определяющая разлагаемую часть опада, для органики, [dim]
    real :: fdpm2 = 0.49                               !< Функция определяющая разлагаемую часть опада, для навоза, [dim]
    ! ------- Coefficients ------------
    real :: ks(ntiles) = (/ 3.22*(1.E-7), 9.65*(1.E-9), 2.12*(1.E-8), 6.43*(1.E-10)/)
    !< Скорость дыхания единицы массы каждого пула в стандартных условиях, [1-s]
    real, parameter :: alphadr = 1.44        !< Определяет соотношение поступления опада между DPM и RPM, [dim],
    ! Для деревьев, для кустарников, для натуральной травы, для СХ (/ 0.25, 0.33, 0.67, 1.44 /)
    
    real, allocatable :: SA(:,:)       !< Определяет распределение выдыхаемой части пула по остальным пулам
    real, allocatable :: CO2_prop(:,:) !< Определяет распределение выдыхаемой части пула в CO2
    real, allocatable :: BIO_prop(:,:) !< Определяет распределение выдыхаемой части пула в CBIO
    real, allocatable :: HUM_prop(:,:) !< Определяет распределение выдыхаемой части пула в CHUM
    
    ! ------ Climate variables --------
    ! ------ Determined within
    real, allocatable :: s0(:,:,:)           !< Оптимальная влажность почвы, [dim]
    real, allocatable :: smin(:,:,:)         !< Переменные для функций, [dim]
    ! ------ Determined externally
    !real :: bettar(i0:i1,j0:j1)             !< Fraction of soil respiration, [%] from 0 to 1
    !real :: sw                              !< Влажность увядания
    ! ------ Initialization ------
    real :: cs = 0.                          !< Сумма пулов
    ! ----- The topsoil moisture deficit -------
    real :: deep = 20.                      ! глубина слоя, см (@todo сопоставить с переменной Ml)
    real, allocatable :: TSMD      (:,:)    !< the topsoil moisture deficit
    real, allocatable :: SMD       (:,:)    !< TSMD на актуальной глубине
    real, allocatable :: bSMD      (:,:)    !< Случай TSMD для голой почвы (bare soil)
    real, allocatable :: moi_bal   (:,:)    !< pr minus evaporation
    real, allocatable :: aTSMD     (:,:)    !< accumulated topsoil mositure deficit 
    real, allocatable :: evpar_part(:,:)    !< 75% from evaporation

    real :: a,b


contains

    subroutine carbon_model_init()

    integer :: i !< count

      allocate(s0(i0:i1,j0:j1,ml))
      allocate(smin(i0:i1,j0:j1,ml))

      allocate(SA(i0:i1,j0:j1))
      allocate(CO2_prop(i0:i1,j0:j1))
      allocate(BIO_prop(i0:i1,j0:j1))
      allocate(HUM_prop(i0:i1,j0:j1))

      ! init of TPMD
      allocate(TSMD(i0:i1,j0:j1))
      allocate(SMD(i0:i1,j0:j1))
      allocate(bSMD(i0:i1,j0:j1))
      allocate(moi_bal(i0:i1,j0:j1))
      allocate(aTSMD(i0:i1,j0:j1))
      allocate(evpar_part(i0:i1,j0:j1))

      TSMD = 0.
      SMD = 0.
      bSMD = 0.
      moi_bal = 0.
      aTSMD = 0.
      evpar_part = 0.
     
      SA = 0.
      CO2_prop = 0.
      BIO_prop = 0.
      HUM_prop = 0.
      
      s0 = 0.
      smin = 0.

      TSMD(:,:) = -(20. + 1.3*bettar(:,:) - 0.01*bettar(:,:)**2)
      SMD(:,:) = (deep/23.)*TSMD(:,:)
      bSMD(:,:) = SMD(:,:)/1.8

     ! main init part of Rothc
      do i = 1, ml
        s0  (:,:,i) = 0.5*(1. + sw(:,:))
        smin(:,:,i) = 1.7*sw(:,:)
      end do

      fdpm = alphadr/(1 + alphadr)

      SA(:,:) = 1.67*(1.85 + 1.6*exp(-0.0786*bettar(:,:)))
      CO2_prop(:,:) = SA(:,:)/(SA(:,:) + 1.)
      BIO_prop(:,:) = 0.46/(SA(:,:) + 1.)
      HUM_prop(:,:) = 0.54/(SA(:,:) + 1.)

    end subroutine



    subroutine carbon_model_calc_at_timestep()


    end subroutine



    subroutine carbon_model_calc_at_cell(ii,jj)

      integer, intent(in) :: ii, jj

     ! ---- TPMD_calc_at_cell ----

      evpar_part(ii,jj) = 0.75*Evpar(ii,jj)
      moi_bal(ii,jj) = pr(ii,jj) - evpar_part(ii,jj)

      if (veg(ii,jj) == 1) then
        aTSMD(ii,jj) = max(SMD(ii,jj), min(0.,aTSMD(ii,jj) + moi_bal(ii,jj)))
      else
        aTSMD(ii,jj) = max(min(bSMD(ii,jj),aTSMD(ii,jj)),min(0.,aTSMD(ii,jj) + moi_bal(ii,jj)))
      end if

     ! main part of rothc_calc_at_cell

      !if (Wsoil(ii,jj,1) >= s0(ii,jj,1)) then
     !   Fs = 1. - 0.8*(Wsoil(ii,jj,1) - s0(ii,jj,1))
     ! else if (smin(ii,jj,1) < Wsoil(ii,jj,1) .and. Wsoil(ii,jj,1) < s0(ii,jj,1)) then
     !   Fs = 0.2 + 0.8*(Wsoil(ii,jj,1) - smin(ii,jj,1))/(s0(ii,jj,1) - smin(ii,jj,1))
     ! else if (Wsoil(ii,jj,1) <= smin(ii,jj,1)) then
     !   Fs = 0.2
     ! end if

      !a = Fs

      if (aTSMD(ii,jj) > 0.444*SMD(ii,jj)) then
        Fs = 1.
      else
        Fs = 0.2 + (1. - 0.2)*(SMD(ii,jj) - aTSMD(ii,jj))/(SMD(ii,jj) - 0.444*SMD(ii,jj))
      end if

      !b = Fs

      if (Tsoil(ii,jj,1) >= -5.) then
        Ft = 47.91/(1. + exp(106.06/(Tsoil(ii,jj,1) + 18.27)))
      else
        Ft = 0
      end if

      ! ---- Initialization ------
      Cs = CDPM(ii,jj,1) + CRPM(ii,jj,1) + CBIO(ii,jj,1) + CHUM(ii,jj,1)

        print*, 'moi_bal:', moi_bal(ii,jj)
        print*, 'pr', pr(ii,jj)
        print*, 'Evpar:', Evpar(ii,jj)
        print*, 'evpar_part:', evpar_part(ii,jj)


    end subroutine



    subroutine carbon_model_calc_at_tile(ii,jj,nn)

        integer, intent(in) :: ii, jj, nn



    end subroutine



    subroutine carbon_model_postprocessing() !Вывод переменных

        use grid, only : date_c

        open(unit=500, file='results/'//trim(carbon_model_type)//'/'//trim(station_name)//'_'//trim(station_opt)//'.txt', status='unknown')
        write(500,*) date_c%timestamp,';',CDPM(:,:,1),';',CRPM(:,:,1),';',CBIO(:,:,1),';',CHUM(:,:,1),';', CIOM(:,:,1)

        write(513,*) date_c%timestamp,';',a,';',b

        ! ---- Initialization ------
        !print*, 'finish thing 1, ', CDPM(:,:,1)/cs*100,'%'
        !print*, 'finish thing 2, ', CRPM(:,:,1)/cs*100,'%'
        !print*, 'finish thing 3, ', CBIO(:,:,1)/cs*100,'%'
        !print*, 'finish thing 4, ', CHUM(:,:,1)/cs*100,'%'
    end subroutine


end module
