module carbon_model_inmcm_aux

    ! интерфейс
! ----------------------------------------------- Use pack ------------------------------------------------------------
    use environment_core, only: Tsoil, Wsoil, lambd
	use config, only: station_name, station_opt, carbon_model_type
    use const,  only: yrs, pi, nmonth
    use grid,   only: date_c
    use environment_model_inmcm, only: nv2
! ---------------------------------------------- Variables ------------------------------------------------------------
    implicit none

    ! ------- Pools -------
    real, dimension(:,:,:), pointer :: Catm    !< атмосфера
    real, dimension(:,:,:), pointer :: Cveg    !< растительность
    real, dimension(:,:,:), pointer :: Csoil   !< почва
    real, dimension(:,:,:), pointer :: Csoilb  !< почва типа b
    ! ---- Flows ----
    real, dimension(:,:,:), pointer :: Fmicr   !< микробное дыхание
    real, dimension(:,:,:), pointer :: Fmicrb  !< микробное дыхание b
    real, dimension(:,:,:), pointer :: Flit    !< отмирание растений
    real, dimension(:,:,:), pointer :: Fers    !< почвенная эрозия
    ! ---- Constant from main program INMCM ----
    real, parameter :: amrp(nv2) = (/0.5, 0.4000, 0.3864, 0.25, 0.25, 0.1250, 0.1700, 0.1909, 0.1909, 0.05, 0.05, 0.2273, 0.2273/)  
                                               !< microbial respiration parameter, [umol co2/kg c/s]
    real, parameter :: tmin(nv2) = (/ 5.,     0.,   -2.5,  -5.,  -5.,     5.,     0.,    -5.,     0.,   0.,   0.,     0.,    0. /)  
                                               !< minimum temperature for photosynthesis, [C]
    real, parameter :: tmin_soil = 0.          !< эквивалент label='fst'
    real, parameter :: t_ref = 25.             !< референсная температура для величин типа rmf25, kc25 и тд [C]
    real, parameter :: t_ref_soil = 10.        !< референсная температура для величин типа Ts [C]
    real, parameter :: hint = 100.             !< Depth above which the total (rswa) and average (rsw) soil moisture is calculated,
                                               ! в модели изначально 200.
    real, parameter :: cv81b = 50.             !< Decomposition time, [year]
    ! ---- Landuse ----
    real, parameter :: gt2year_to_kg2s = 1.e+12 / yrs
    real, parameter :: defor0 = 1. * gt2year_to_kg2s
    !real, parameter :: defor1861 = 0.54 * gt2year_to_kg2s
    real, parameter :: adefr  = 0.6                 !< deforest/(deforest+soil erosion)
    real, parameter :: defr(nv2) = (/1.,0.2,0.2,0.05,0.05,0.2,0.2,0.2,0.02,0.02,0.2,0.5,0.5/)
    real, parameter :: soer(nv2) = (/1.,0.2,0.2,0.05,0.05,0.2,0.2,0.2,0.02,0.02,0.2,0.5,0.5/)
    real :: landuseErs                              !< замена переменных для defor
    real, allocatable, target :: ers_weight(:,:,:)  !< пространственные веса
    real :: conversion_defor_global_to_regional
    ! ---- Динамические переменные (from inmcm) ----
    real, target :: rsw(nv2)                        !< Показатель влажности на глубине
contains

    subroutine carbon_model_init()

        use netcdf
        use grid,       only : i0, i1, j0, j1, area, date_c, dlon, dlat, nlon, nlat, ich
        use netcdf_kit, only : nc_errhand
        use paths,      only : path_inmcm_data

        integer :: i, j, n      !< count
        integer :: ncid, varid
        
        real, allocatable, dimension(:,:,:) :: cveg_global, csoil_global
        real :: sc6_global, sc6_regional
        real :: sc8_global, sc8_regional

        allocate(ers_weight(i0:i1,j0:j1,nv2))

        ! ---- Variables from INMCM ----

        if (dlon == 0.5 .and. dlat == 0.5) then

            allocate(cveg_global(nlon,nlat,nv2))
            allocate(csoil_global(nlon,nlat,nv2))
            
            call nc_errhand( nf90_open(path_inmcm_data//'control_point_for_carbon_pools.nc', nf90_nowrite, ncid) )
            call nc_errhand( nf90_inq_varid(ncid, 'cveg', varid) )
            call nc_errhand( nf90_get_var(ncid, varid, cveg_global(:,:,:), (/1,1,1/), (/nlon,nlat,nv2/)) )
            call nc_errhand( nf90_inq_varid(ncid, 'csoil', varid) )
            call nc_errhand( nf90_get_var(ncid, varid, csoil_global(:,:,:), (/1,1,1/), (/nlon,nlat,nv2/)) )
            call nc_errhand( nf90_close(ncid) )

            sc6_global = 0.
            sc8_global = 0.
            do j = 1, nlat          ! sum for the whole globe
                do i = 1, nlon
                    if (ich(i,j) /= 0) then
                        do n = 1, nv2
                            sc6_global = sc6_global + cveg_global(i,j,n)*defr(n)*area(i,j)
                            sc8_global = sc8_global + csoil_global(i,j,n)*soer(n)*area(i,j)
                        enddo
                    endif
                enddo
            enddo
            
            if (sc6_global > 0. .and. sc8_global > 0.) then
                sc6_regional = 0.
                sc8_regional = 0.
                do j = j0, j1          ! sum for the regional simulation domain
                    do i = i0, i1
                        if (ich(i,j) /= 0) then
                            do n = 1, nv2
                                sc6_regional = sc6_regional + cveg_global(i,j,n)*defr(n)*area(i,j)
                                sc8_regional = sc8_regional + csoil_global(i,j,n)*soer(n)*area(i,j)
                            enddo
                        endif
                    enddo
                enddo
                conversion_defor_global_to_regional = adefr * sc6_regional/sc6_global + &
                                               & (1.-adefr) * sc8_regional/sc8_global
            else
                conversion_defor_global_to_regional = 1.
            endif

			deallocate(cveg_global)
            deallocate(csoil_global)
            
        else
            conversion_defor_global_to_regional = 1.
        endif

        landuseErs =  0. !(1. - adefr) * defor0 * conversion_defor_global_to_regional

    end subroutine



    subroutine carbon_model_calc_at_timestep()

        use grid, only : i0, i1, j0, j1, area

        integer :: ii, jj, n

        real :: sc8

        sc8 = 0.
        do ii = i0, i1
            do jj = j0, j1
                do n = 1, nv2
                    sc8 = sc8 + Csoil(ii,jj,n) * soer(n) * area(ii,jj)
                end do
            end do
        end do

        ers_weight(:,:,:) = 0.
        if (sc8 > 0) then
            do ii = i0, i1
                do jj = j0, j1
                    do n = 1, nv2
                        ers_weight(ii,jj,n) = Csoil(ii,jj,n) * soer(n) / sc8
                    end do
                end do
            end do
        end if

    end subroutine


    subroutine carbon_model_calc_at_cell(ii,jj)

        integer, intent(in) :: ii, jj


    end subroutine



    subroutine carbon_model_calc_at_tile(ii,jj,nn)

        use grid, only : ms, ml, z
        use environment_core, only : g2gw_to_cmw

        integer, intent(in) :: ii, jj, nn
        integer :: k
        real :: work

            work = 0.
            do k = ms+1, ml-1
                if (z(k) <= hint) then
                    work = work + Wsoil(ii,jj,k) * g2gw_to_cmw(ii,jj,k)
                end if
            end do
            rsw(nn) = work / hint

    end subroutine



    subroutine carbon_model_postprocessing()

        use grid, only : date_c

        open(unit=500, file='results/'//trim(carbon_model_type)//'/'//trim(station_name)//'_'//trim(station_opt)//'.txt', status='unknown')
        write(500,*) date_c%timestamp,';',Csoil(:,:,12),';',Csoilb(:,:,12)

    end subroutine


end module
