
module carbon_model_to_core_fun_kit

    !< @brief библиотека мультипликативных функций

    ! интерфейс
    ! ---------------------------------------------------------------------------------
    
    use carbon_core, only : fun_kit_type, narg_default, npar_default

    implicit none
    
    private
    public :: set_fun
    public :: get_fun_result
    
    
contains
    
    
    ! внешние процедуры
    ! ---------------------------------------------------------------------------------
    
    subroutine set_fun(fun_kit, funtype)
        ! ---------------------------------------
        !< @brief установка вида функций и их параметров
        
        type(fun_kit_type), intent(inout) :: fun_kit
        character(*), intent(in) :: funtype
        
        select case (funtype)
            case('const')
                fun_kit%fun => fun_const
            case('lin')
                fun_kit%fun => fun_lin
            case('exp')
                fun_kit%fun => fun_exp
            case('hyp')
                fun_kit%fun => fun_hyp
            case('mm')
                fun_kit%fun => fun_mm
            case('step')
                fun_kit%fun => fun_step
            !case('spline')
                !fun_kit%fun => fun_spline
            case default
                stop "check failed : unknown funtype at set_fun"
        end select
        
    end subroutine
    
    
    function get_fun_result(fun_kit, args, pars) result (mult)
        ! ---------------------------------------
        type(fun_kit_type), intent(in) :: fun_kit
        real, intent(in) :: args(narg_default), pars(npar_default)
        real :: mult
        
        mult = fun_kit%fun(args, pars)
        
    end function
    
    
    ! внутренние процедуры
    ! ---------------------------------------------------------------------------------

    ! args(1) - x
    ! pars(1) - y0 или c
    ! pars(2) - x0
    ! pars(3) - k
    ! pars(4) - a
    ! pars(5) - amp

    !> Constant function: y = c
    function fun_const(args, pars) result(y)
        implicit none
        real, intent(in) :: args(narg_default)
        real, intent(in) :: pars(npar_default)
        real :: y
        y = pars(1)
    end function fun_const

    !> Linear function: y = k*(x-x0) + y0
    function fun_lin(args, pars) result(y)
        implicit none
        real, intent(in) :: args(narg_default)
        real, intent(in) :: pars(npar_default)
        real :: y
        y = pars(3) * (args(1) - pars(2)) + pars(1)
    end function fun_lin

    !> Hyperbolic function: y = k/(x-x0) + y0
    function fun_hyp(args, pars) result(y)
        implicit none
        real, intent(in) :: args(narg_default)
        real, intent(in) :: pars(npar_default)
        real :: y
        y = pars(3) / (args(1) - pars(2)) + pars(1)
    end function fun_hyp

    !> Michaelis-Menthen function: y = k*x/(x-x0) + y0
    function fun_mm(args, pars) result(y)
        implicit none
        real, intent(in) :: args(narg_default)
        real, intent(in) :: pars(npar_default)
        real :: y
        y = pars(3) * args(1) / (args(1) - pars(2)) + pars(1)
    end function fun_mm

    !> Step function: y = k*θ(x-x0) + y0
    function fun_step(args, pars) result(y)
        implicit none
        real, intent(in) :: args(narg_default)
        real, intent(in) :: pars(npar_default)
        real :: y, theta
        theta = 0.5*(1. + sign(1.,args(1) - pars(2)))
        y = pars(1) + pars(3) * theta
    end function fun_step

    !> Exponent function: y = amp*a**[k*(x-x0)] + y0
    function fun_exp(args, pars) result(y)
        implicit none
        real, intent(in) :: args(narg_default)
        real, intent(in) :: pars(npar_default)
        real :: y
        y = pars(5) * pars(4)**( pars(3) * (args(1) - pars(2)) ) + pars(1)
    end function fun_exp
    
    !> Spline function: y = (a+k)*x + (y0+c)/2 + (a-k)/2*|x-x0|
    function fun_spline(args, pars) result(y)
        implicit none
        real, intent(in) :: args(narg_default)
        real, intent(in) :: pars(npar_default)
        real :: y
        y = (pars(4)+pars(3))*args(1) + (pars(1)+pars(5))/2 + (pars(4)-pars(3))/2*abs(args(1)-pars(2))
    end function fun_spline 
    

end module
