module carbon

    !< @brief модуль углерода
    !< @details главный модуль, маршрутизация вызовов подпрограмм

    use config, only : carbon_model_type, &
                    & if_standard_print, &
                    & if_standard_output, &
                    & testing_log_mode
    use carbon_solver, only : carbon_solver_fluxes, &
                            & carbon_solver_pools
    use carbon_postprocessing, only : carbon_standard_print, &
                                    & carbon_standard_output, &
                                    & carbon_testing_write_log, &
                                    & carbon_testing_read_log
    use carbon_model_inmcm, only : carbon_model_assembly_inmcm => carbon_model_assembly
    use carbon_model_inmcm_aux, only : carbon_model_init_inmcm => carbon_model_init, &
                                     & carbon_model_calc_at_timestep_inmcm => carbon_model_calc_at_timestep, &
                                     & carbon_model_calc_at_cell_inmcm => carbon_model_calc_at_cell, &
                                     & carbon_model_calc_at_tile_inmcm => carbon_model_calc_at_tile, &
                                     & carbon_model_postprocessing_inmcm => carbon_model_postprocessing
    use carbon_model_socs, only : carbon_model_assembly_socs => carbon_model_assembly
    use carbon_model_socs_aux, only : carbon_model_init_socs => carbon_model_init, &
                                    & carbon_model_calc_at_timestep_socs => carbon_model_calc_at_timestep, &
                                    & carbon_model_calc_at_cell_socs => carbon_model_calc_at_cell, &
                                    & carbon_model_calc_at_tile_socs => carbon_model_calc_at_tile, &
                                    & carbon_model_postprocessing_socs => carbon_model_postprocessing
    use carbon_model_rothc, only : carbon_model_assembly_rothc => carbon_model_assembly
    use carbon_model_rothc_aux, only : carbon_model_init_rothc => carbon_model_init, &
                                    & carbon_model_calc_at_timestep_rothc => carbon_model_calc_at_timestep, &
                                    & carbon_model_calc_at_cell_rothc => carbon_model_calc_at_cell, &
                                    & carbon_model_calc_at_tile_rothc => carbon_model_calc_at_tile, &
                                    & carbon_model_postprocessing_rothc => carbon_model_postprocessing
    use carbon_model_user, only : carbon_model_assembly_user => carbon_model_assembly
    use carbon_model_user_aux, only : carbon_model_init_user => carbon_model_init, &
                                        & carbon_model_calc_at_timestep_user => carbon_model_calc_at_timestep, &
                                        & carbon_model_calc_at_cell_user => carbon_model_calc_at_cell, &
                                        & carbon_model_calc_at_tile_user => carbon_model_calc_at_tile, &
                                        & carbon_model_postprocessing_user => carbon_model_postprocessing
    
    implicit none

    private
    public :: carbon_init
    public :: carbon_calc_at_timestep
    public :: carbon_calc_at_cell
    public :: carbon_calc_at_tile
    public :: carbon_postprocessing

    interface
        subroutine interface_init()
        end subroutine
        subroutine interface_assembly()
        end subroutine
        subroutine interface_calc_at_timestep()
        end subroutine
        subroutine interface_calc_at_cell(ii,jj)
            integer, intent(in) :: ii, jj
        end subroutine
        subroutine interface_calc_at_tile(ii,jj,nn)
            integer, intent(in) :: ii, jj, nn
        end subroutine
        subroutine interface_postprocessing()
        end subroutine
    end interface
    
    procedure(interface_init), pointer :: carbon_model_init
    procedure(interface_assembly), pointer :: carbon_model_assembly
    procedure(interface_calc_at_timestep), pointer :: carbon_model_calc_at_timestep
    procedure(interface_calc_at_cell), pointer :: carbon_model_calc_at_cell
    procedure(interface_calc_at_tile), pointer :: carbon_model_calc_at_tile
    procedure(interface_postprocessing), pointer :: carbon_model_postprocessing

contains


    subroutine carbon_init()
    
        select case(carbon_model_type)
            case('inmcm')
                carbon_model_init => carbon_model_init_inmcm
                carbon_model_assembly => carbon_model_assembly_inmcm
                carbon_model_calc_at_timestep => carbon_model_calc_at_timestep_inmcm
                carbon_model_calc_at_cell => carbon_model_calc_at_cell_inmcm
                carbon_model_calc_at_tile => carbon_model_calc_at_tile_inmcm
                carbon_model_postprocessing => carbon_model_postprocessing_inmcm
            case('rothc')
                carbon_model_init => carbon_model_init_rothc
                carbon_model_assembly => carbon_model_assembly_rothc
                carbon_model_calc_at_timestep => carbon_model_calc_at_timestep_rothc
                carbon_model_calc_at_cell => carbon_model_calc_at_cell_rothc
                carbon_model_calc_at_tile => carbon_model_calc_at_tile_rothc
                carbon_model_postprocessing => carbon_model_postprocessing_rothc
            case('socs')
                carbon_model_init => carbon_model_init_socs
                carbon_model_assembly => carbon_model_assembly_socs
                carbon_model_calc_at_timestep => carbon_model_calc_at_timestep_socs
                carbon_model_calc_at_cell => carbon_model_calc_at_cell_socs
                carbon_model_calc_at_tile => carbon_model_calc_at_tile_socs
                carbon_model_postprocessing => carbon_model_postprocessing_socs
            case('other')
                carbon_model_init => carbon_model_init_user
                carbon_model_assembly => carbon_model_assembly_user
                carbon_model_calc_at_timestep => carbon_model_calc_at_timestep_user
                carbon_model_calc_at_cell => carbon_model_calc_at_cell_user
                carbon_model_calc_at_tile => carbon_model_calc_at_tile_user
                carbon_model_postprocessing => carbon_model_postprocessing_user
        end select
        
        call carbon_model_init()
        
        call carbon_model_assembly()
        
        if (if_standard_output) call carbon_standard_output()
        
    end subroutine


    subroutine carbon_calc_at_timestep()
        
        call carbon_model_calc_at_timestep()
        
    end subroutine


    subroutine carbon_calc_at_cell(ii,jj)
    
        integer, intent(in) :: ii, jj
        
        call carbon_model_calc_at_cell(ii,jj)
        
    end subroutine


    subroutine carbon_calc_at_tile(ii,jj,nn)
    
        integer, intent(in) :: ii, jj, nn
        
        call carbon_model_calc_at_tile(ii,jj,nn)
        
        call carbon_solver_fluxes(ii,jj,nn)
        
        call carbon_solver_pools(ii,jj,nn)
        
    end subroutine


    subroutine carbon_postprocessing()
        
        select case(testing_log_mode)
            case('write')
                call carbon_testing_write_log()
            case('read')
                call carbon_testing_read_log()
            case('none')
        end select
        
        if (if_standard_print) call carbon_standard_print()
        
        if (if_standard_output) call carbon_standard_output()
        
        call carbon_model_postprocessing()
        
    end subroutine


end module
