MODULE DIFFUSION_MOD

implicit none

contains

!> Subroutine DIFFUSION performs one timestep in diffusion
!! using tetha scheme
SUBROUTINE DIFFUSION(N, bctype, bcs, dt, ddz, var1, var2, diff, tetha)
! Modules
use NUMERICS, only : PROGONKA
implicit none
!Input variables
integer, intent(in) :: N            !< Number of levels of finite-difference grid
real   , intent(in) :: tetha        !< Weighting factor
integer, intent(in) :: bctype(1:2)  !< Kind of boundary conditions
real   , intent(in) :: dt           !< Timestep
real   , intent(in) :: bcs(1:2)     !< Value of variable or flux at boundaries
real   , intent(in) :: diff(1:N-1)  !< Diffusion coefficient
real   , intent(in) :: ddz(1:N + 1) !< Grid spacing
real   , intent(in) :: var1(1:N)    !< Variable to be diffused at the previous timestep
!Output variables
real   , intent(out) :: var2(1:N)  !< Variable to be diffused at the next timestep:w

!Local variables
real :: dz1, dz2, alpha

real, allocatable :: a(:), b(:), c(:), f(:), y(:)

integer :: i !Loop index

allocate (a(1:N),b(1:N),c(1:N), &
&         f(1:N),y(1:N))
a(:) = 0.; b(:) = 0.; c(:) = 0.; f(:) = 0.

!Top boundary conditions
if     (bctype(1) == 1) then !Dirichlet condition
  c(1) = 1.
  b(1) = 0.
  f(1) = bcs(1)
elseif (bctype(1) == 2) then !Neumann condition
  dz1 = ddz(2)
  alpha = (2 * dt) / (ddz(2) + ddz(1))
  b(1)   = alpha * tetha * diff(1) / dz1
  c(1)   = 1 + b(1)
  f(1)   = (1 - alpha * (1 - tetha) * diff(1) / dz1) * var1(1) + (alpha * (1 - tetha) * diff(1) / dz1) * var1(2) - &
          & bcs(1) * alpha
endif

!Coefficients for domain interior
do i = 2, N-1

  dz1 = ddz(i + 1)
  dz2 = ddz(i)
  alpha = (2 * dt) / (dz1 + dz2)

  a(i) = alpha * tetha * diff(i - 1) / dz2
  b(i) = alpha * tetha * diff(i) / dz1
  c(i) = b(i) + a(i) + 1

  f(i) = (alpha * (1 - tetha) * diff(i - 1) / dz2) * var1(i - 1) + &
  & (1 - alpha * (1 - tetha) * diff(i) / dz1 - alpha * (1 - tetha) * diff(i - 1) / dz2) * var1(i) + &
  & (alpha * (1 - tetha) * diff(i) / dz1) * var1(i + 1)


enddo

!Bottom boundary conditions
if     (bctype(2) == 1) then !Dirichlet condition
  c(N) = 1.
  a(N) = 0.
  f(N) = bcs(2) 
elseif (bctype(2) == 2) then !Neumann condition
  dz2 = ddz(N)
  alpha = (2 * dt) / (ddz(N + 1) + ddz(N))
  a(N) = alpha * tetha * diff(N - 1) / dz2
  c(N) = 1 + a(N)
  f(N) = alpha * bcs(2) - (alpha * (1 - tetha) * diff(N - 1) / dz2) * (var1(N) - var1(N - 1)) + var1(N)
endif

call PROGONKA (N, a, b, c, f, var2, 1, N)
deallocate (a, b, c, f)
END SUBROUTINE DIFFUSION

END MODULE DIFFUSION_MOD
