#pragma once
#include <vector>
#include <map>
#include <string>
#include <utility>
#include <numeric>
#include <iostream>
#include <fstream>
#include <sstream>
#include <iomanip>
#include <memory>

#ifdef USE_JSON_OUTPUT
        #include <nlohmann/json.hpp>
        using json = nlohmann::json;
#endif

#include "TemplateParameters.h"
#include "defines.hpp"

namespace icethermo
{
    template <typename NumType, MemType memtype>
    class Mesh
    {
    public:
        // Constructors
        Mesh(); // default constructor for empty mesh
        Mesh(NumType thickness); // constructor with 10 uniform layers with given thicknes
        Mesh(int n_uniform_layers, NumType thickness); // constructor with given number of uniform layers and given thickness
        Mesh(const NumType* unit_segment_decomposition, const int unit_segment_decomposition_size, NumType thickness); // constructor with manual partition for sigma layer grid and thickness
        Mesh(const Mesh<NumType, memtype>& other); // copy constructor
        ~Mesh(); // destructor

//         // cells and nodes number getters
        int GetCellsNum() const;
        int GetNodesNum() const;

//         // Creators of single, cells and nodes data
        std::shared_ptr<NumType> CreateSingleData(const std::string& varname, bool visible = true);
        std::shared_ptr<NumType*> CreateCellsData(const std::string& varname, bool visible = true);
        std::shared_ptr<NumType*> CreateNodesData(const std::string& varname, bool visible = true);

//         // Deleters of single, cells and nodes data
        void DeleteSingleData(const std::string& varname);
        void DeleteCellsData(const std::string& varname);
        void DeleteNodesData(const std::string& varname);

//         // Getters of single, cells and nodes data
        std::shared_ptr<NumType> GetSingleData(const std::string& varname);
        std::shared_ptr<NumType*> GetCellsData(const std::string& varname);
        std::shared_ptr<NumType*> GetNodesData(const std::string& varname);

//         // Getter for cell thicknesses and total thickness
        std::shared_ptr<NumType*> GetCellsThickness();
        NumType GetTotalThickness() const;

//         // Muters and Unmuters
        void MuteSingleData(const std::string& varname);
        void MuteCellData(const std::string& varname);
        void MuteNodeData(const std::string& varname);
        
        void UnmuteSingleData(const std::string& varname);
        void UnmuteCellData(const std::string& varname);
        void UnmuteNodeData(const std::string& varname);

//         // Write mesh to File
        void SaveTXT(const std::string& filename);
        void SaveTXT(const std::string& filename, int postscript);

#ifdef USE_JSON_OUTPUT
        // void SaveJSON(const std::string& filename);
        // void SaveJSON(const std::string& filename, int postscript);
#endif

        // Check existency of data
        bool CheckCellsDataExistency(const std::string& varname) const;
        bool CheckNodesDataExistency(const std::string& varname) const;
        bool CheckSingleDataExistency(const std::string& varname) const;

        // Check GPU results subtools
        void PullCPUArray(const NumType* array, const int n);

    private:
        // vector of cell thicknesses
        std::shared_ptr<NumType*> cells_thickness;
        size_t cells_thickness_size_t;
        // container for stand-alone variables
        std::map<std::string, std::pair<std::shared_ptr<NumType>, bool>> single_data;

        // container for cell data [name : {vector_of_values, is_visible}]
        std::map<std::string, std::pair<std::shared_ptr<NumType*>, bool>> cells_data;
        std::map<std::string, size_t> cells_data_size_t;

        // container for node data [name : {vector_of_values, is_visible}]
        std::map<std::string, std::pair<std::shared_ptr<NumType*>, bool>> nodes_data;
        std::map<std::string, size_t> nodes_data_size_t;

     // sub-bufer for data output (needs if data on the GPU)
        NumType* SubBuffer;
        size_t SubBuffer_size;
    };
}